using System.Collections.Generic;
using System.Linq;
using UnityEngine.Experimental.VFX;

namespace UnityEditor.VFX.Block
{
    enum CameraMode
    {
        Main,
        Custom,
    }

    struct CameraMatricesExpressions
    {
        public VFXNamedExpression ViewToVFX;
        public VFXNamedExpression VFXToView;
        public VFXNamedExpression ViewToClip;
        public VFXNamedExpression ClipToView;
    }

    static class CameraHelper
    {
        public class CameraProperties
        {
            public CameraType Camera = CameraType.defaultValue;
        }

        public static IEnumerable<VFXNamedExpression> AddCameraExpressions(IEnumerable<VFXNamedExpression> expressions, CameraMode mode)
        {
            if (mode == CameraMode.Main)
            {
                expressions = expressions.Append(new VFXNamedExpression(new VFXExpressionExtractMatrixFromMainCamera(), "Camera_transform"));
                expressions = expressions.Append(new VFXNamedExpression(new VFXExpressionExtractFOVFromMainCamera(), "Camera_fieldOfView"));
                expressions = expressions.Append(new VFXNamedExpression(new VFXExpressionExtractNearPlaneFromMainCamera(), "Camera_nearPlane"));
                expressions = expressions.Append(new VFXNamedExpression(new VFXExpressionExtractFarPlaneFromMainCamera(), "Camera_farPlane"));
                expressions = expressions.Append(new VFXNamedExpression(new VFXExpressionExtractAspectRatioFromMainCamera(), "Camera_aspectRatio"));
                expressions = expressions.Append(new VFXNamedExpression(new VFXExpressionExtractPixelDimensionsFromMainCamera(), "Camera_pixelDimensions"));
                expressions = expressions.Append(new VFXNamedExpression(new VFXExpressionGetBufferFromMainCamera(VFXCameraBufferTypes.Depth), "Camera_depthBuffer"));
                expressions = expressions.Append(new VFXNamedExpression(new VFXExpressionGetBufferFromMainCamera(VFXCameraBufferTypes.Color), "Camera_colorBuffer"));
            }

            return expressions;
        }

        public static CameraMatricesExpressions GetMatricesExpressions(IEnumerable<VFXNamedExpression> expressions)
        {
            var fov = expressions.First(e => e.name == "Camera_fieldOfView");
            var aspect = expressions.First(e => e.name == "Camera_aspectRatio");
            var near = expressions.First(e => e.name == "Camera_nearPlane");
            var far = expressions.First(e => e.name == "Camera_farPlane");
            var cameraMatrix = expressions.First(e => e.name == "Camera_transform");

            VFXExpression ViewToVFX = cameraMatrix.exp;
            VFXExpression VFXToView = new VFXExpressionInverseTRSMatrix(ViewToVFX);
            VFXExpression ViewToClip = VFXOperatorUtility.GetPerspectiveMatrix(fov.exp, aspect.exp, near.exp, far.exp);
            VFXExpression ClipToView = new VFXExpressionInverseMatrix(ViewToClip);

            return new CameraMatricesExpressions()
            {
                ViewToVFX = new VFXNamedExpression(ViewToVFX, "ViewToVFX"),
                VFXToView = new VFXNamedExpression(VFXToView, "VFXToView"),
                ViewToClip = new VFXNamedExpression(ViewToClip, "ViewToClip"),
                ClipToView = new VFXNamedExpression(ClipToView, "ClipToView"),
            };
        }
    }
}
